//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for basic QuickTime movie display and control.
//				This file is used for BOTH MacOS and Windows.
//
//	Written by:	Tim Monroe
//				Based on the MovieShell code written by Apple DTS.
//
//	Copyright:	 1994-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <18>	 	12/16/98	rtm		removed all QTVR API calls, so we can remove QTVR.lib from project
//	   <17>	 	05/21/98	rtm		removed conditionalized support for compound effects
//	   <16>	 	02/20/98	rtm		revised custom dialog box handling; now works on Windows (yippee!)
//	   <15>	 	02/12/98	rtm		added support for stepping through the effect
//	   <14>	 	02/05/98	rtm		reworked HandleContentClick and DoApplicationEventLoopAction
//	   <13>	 	12/18/97	rtm		added low-level effects calls for QTShowEffect
//	   <12>	 	11/21/97	rtm		added code to DoApplicationEventLoopAction to handle effects dialog events
//	   <11>	 	11/06/97	rtm		removed QTVR support; added support for MakeEffectMovie routines
//	   <10>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <9>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu identifiers
//	   <8>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <7>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <6>	 	06/04/97	rtm		removed call to QTVRUtils_IsQTVRMovie in InitApplicationWindowObject
//	   <5>	 	02/06/97	rtm		fixed window resizing code
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//	   
//////////

// header files
#include "ComApplication.h"
#include "QTShowEffect.h"

// global variables for Macintosh code
#if TARGET_OS_MAC
Str255						gAppName = "\pQTShowEffect";		// the name of this application
extern Boolean				gQuitFlag;
#endif

// global variables for Windows code
#if TARGET_OS_WIN32
#endif

long						gMaxMilliSecToUse = 0L;	
	
// external variables
extern QTParameterDialog	gEffectsDialog;
extern DialogPtr			gCustomDialog;
extern StateInformation		gCurrentState;
extern Boolean				gUseStandardDialog;
extern Boolean				gFastEffectDisplay;
extern int					gNumberOfSteps;
extern unsigned short		gLoopingState;
extern unsigned short		gCurrentDir;
extern WindowPtr			gMainWindow;
extern GWorldPtr			gGW1;
extern GWorldPtr			gGW2;
extern GWorldPtr			gGW3;


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// *before* the MDI frame window is created or *after*. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// ***do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {
		// check to make sure that QuickTime video effects are available;
		// we depend on these features  
		if (!QTUtils_HasQuickTimeVideoEffects())
			QuitFramework();
	}

	// ***do any start-up activities that should occur after the MDI frame window is created
	if (theStartPhase & kInitAppPhase_AfterCreateFrameWindow) {
		// initialize for QuickTime effects
		QTEffects_Init();
	}
}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// *before* any open movie windows are destroyed or *after*.
//
//////////

void StopApplication (UInt32 theStopPhase)
{
	if (theStopPhase & kStopAppPhase_BeforeDestroyWindows)
		QTEffects_Stop();
}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif	// TARGET_OS_MAC
		}
	}
	
	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
#pragma unused(theRefreshArea)

	GrafPtr 			mySavedPort;
	WindowPtr			myWindow = GetPortFromWindowReference(theWindow);
	
	GetPort(&mySavedPort);
	MacSetPort((GrafPtr)myWindow);
	// update the main effects window only if we are not showing an effect;
	// otherwise, we will get a flash of the first source image if there is
	// an update while the effect is running
	
	// because the effect updates consecutive frames relatively quickly
	// while it is running, the effect drawing acts as an update scheme anyway
	if (!gCurrentState.fShowingEffect) {
		BeginUpdate(GetPortFromWindowReference(theWindow));
		if (myWindow == gMainWindow)
			// draw the main effects window
			QTEffects_DrawEffectsWindow();
		else	
			// draw the movie controller and its movie
			MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, myWindow);
	
		EndUpdate(GetPortFromWindowReference(theWindow));
	}
	
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
#pragma unused(theEvent)
	WindowPtr			myWindow = GetPortFromWindowReference(theWindow);

	// clicking in the main effects window stops the effect
	if (myWindow == gMainWindow) {
		gCurrentState.fShowingEffect = false;
		gCurrentState.fTime = 1;
		QTEffects_DrawEffectsWindow();
	}
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean		isHandled = true;
	
	switch (theCharCode) {
	
		// @@@HANDLE APPLICATION-SPECIFIC KEY PRESSES HERE

		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	*high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	*low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
	OSErr				myErr = noErr;
	
	switch (theMenuItem) {
		case IDM_SELECT_EFFECT:					// let the user select an effect
		case IDM_COMPOUND_EFFECT:				// add a compound effect to an existing effect
		
			// stop the current effect (if any) from playing
			gCurrentState.fShowingEffect = false;
			gCurrentState.fTime = 0;
			QTEffects_DrawEffectsWindow();
			
			// let the user select an effect
			myErr = QTEffects_LetUserChooseEffect();
			if (myErr != noErr)
				break;

			// if the sample description is already allocated, deallocate it
			if (gCurrentState.fSampleDescription != NULL)
				DisposeHandle((Handle)gCurrentState.fSampleDescription);
				
			// create a sample description for the effect
			gCurrentState.fSampleDescription = QTEffects_MakeSampleDescription(gCurrentState.fEffectType, kWidth, kHeight);
			if (gCurrentState.fSampleDescription == NULL)
				break;
			
			if (theMenuItem == IDM_SELECT_EFFECT) {
				// if the effect description is already allocated, dispose of it
				if (gCurrentState.fEffectDescription != NULL)
					QTDisposeAtomContainer(gCurrentState.fEffectDescription);
				
				// set up a new effect description
				gCurrentState.fEffectDescription = QTEffects_CreateEffectDescription(gCurrentState.fEffectType, kSourceOneName, kSourceTwoName);
				if (gCurrentState.fEffectDescription == NULL)
					break;
					
				// prompt the user to select options
				myErr = QTEffects_LetUserCustomizeEffect(gCurrentState.fEffectDescription);
				if (myErr != noErr)
					break;
				
			} else {
#if ALLOW_COMPOUND_EFFECTS
				// TO BE PROVIDED
#endif
			}
				
			break;

		case IDM_RUN_EFFECT:
			gCurrentState.fShowingEffect = true;
			gCurrentState.fSteppingEffect = false;
			break;

		case IDM_STEP_AHEAD:
			gCurrentState.fShowingEffect = true;
			gCurrentState.fSteppingEffect = true;
			gCurrentDir = kForward;
			gFastEffectDisplay = false;
			break;

		case IDM_STEP_BACK:
			gCurrentState.fShowingEffect = true;
			gCurrentState.fSteppingEffect = true;
			gCurrentDir = kBackward;
			gFastEffectDisplay = false;
			break;

		case IDM_MAKE_EFFECT_MOVIE:
			QTEffects_CreateEffectsMovie(gCurrentState.fEffectType, gCurrentState.fEffectDescription, kWidth, kHeight);
			break;

		case IDM_GET_FIRST_PICTURE:
			myErr = QTEffects_GetPictureAsGWorld(kWidth, kHeight, kDepth, &gGW1);
			if (myErr == noErr) {
				// we need to refresh image descriptions, etc.
				QTEffects_SetUpEffectSequence();
				QTEffects_DrawEffectsWindow();
			}
			break;
			
		case IDM_GET_SECOND_PICTURE:
			myErr = QTEffects_GetPictureAsGWorld(kWidth, kHeight, kDepth, &gGW2);
			if (myErr == noErr) {
				// we need to refresh image descriptions, etc.
				QTEffects_SetUpEffectSequence();
				QTEffects_DrawEffectsWindow();
			}
			break;
			
		case IDM_GET_THIRD_PICTURE:
			myErr = QTEffects_GetPictureAsGWorld(kWidth, kHeight, kDepth, &gGW3);
			if (myErr == noErr) {
				// we need to refresh image descriptions, etc.
				QTEffects_SetUpEffectSequence();
				QTEffects_DrawEffectsWindow();
			}
			break;

		case IDM_NO_LOOPING:
		case IDM_NORMAL_LOOPING:
		case IDM_PALINDROME_LOOPING:
			gLoopingState = theMenuItem;
			gFastEffectDisplay = (theMenuItem == IDM_NO_LOOPING);
			break;
			
		case IDM_STANDARD_DIALOG:
		case IDM_CUSTOM_DIALOG:
			gUseStandardDialog = (theMenuItem == IDM_STANDARD_DIALOG);
			break;
			
		case IDM_FAST_DISPLAY:
			gFastEffectDisplay = !gFastEffectDisplay;
			gLoopingState = kNoLooping;
			break;

		default:
			break;
	} // switch (theMenuItem)
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
#pragma unused(theWindow)
	MenuReference			myMenu;
	
#if TARGET_OS_WIN32
	myMenu = theMenu;
#endif

	// adjust the Settings menu
#if TARGET_OS_MAC
#pragma unused(theMenu)
	myMenu = GetMenuHandle(kSettingsMenu);
#endif

	SetMenuItemCheck(myMenu, IDM_NO_LOOPING, (gLoopingState == kNoLooping));
	SetMenuItemCheck(myMenu, IDM_NORMAL_LOOPING, (gLoopingState == kNormalLooping));
	SetMenuItemCheck(myMenu, IDM_PALINDROME_LOOPING, (gLoopingState == kPalindromeLooping));
	SetMenuItemCheck(myMenu, IDM_STANDARD_DIALOG, (gUseStandardDialog == true));
	SetMenuItemCheck(myMenu, IDM_CUSTOM_DIALOG, (gUseStandardDialog == false));
	SetMenuItemCheck(myMenu, IDM_FAST_DISPLAY, (gFastEffectDisplay == true));

	// now, do all Effects menu adjustment
#if TARGET_OS_MAC
	myMenu = GetMenuHandle(kEffectMenu);
#endif

#if !ALLOW_COMPOUND_EFFECTS
	SetMenuItemState(myMenu, IDM_COMPOUND_EFFECT, kDisableMenuItem);
	SetMenuItemState(myMenu, IDM_GET_THIRD_PICTURE, kDisableMenuItem);
#endif

#if TARGET_OS_MAC
	// we don't allow creating new files here...
	SetMenuItemState(GetMenuHandle(mFile), iNew, kDisableMenuItem);
#endif
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
	Boolean		isHandled = false;
	
	// run the next step(s) of the effect
	QTEffects_ProcessEffect();
	
	// see if the event is meant for the effects parameter dialog box
	if (gEffectsDialog != 0L)
		isHandled = QTEffects_HandleEffectsDialogEvents(theEvent, 0);

#if TARGET_OS_MAC
	// handle events for any application-specific windows
	if (!isHandled)
		isHandled = QTEffects_HandleEffectsWindowEvents(theEvent);
#endif

	return(isHandled);
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{
#pragma unused(theWindowObject)
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
#pragma unused(theWindowObject)
	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	switch (theAction) {
	
		// handle window resizing
		case mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			break;

		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}


